package org.libvirt.jna;

import com.sun.jna.Callback;
import com.sun.jna.Library;
import com.sun.jna.Native;
import com.sun.jna.NativeLong;
import com.sun.jna.Pointer;
import com.sun.jna.ptr.IntByReference;
import com.sun.jna.ptr.LongByReference;

/**
 * The libvirt interface which is exposed via JNA. The complete API is
 * documented at http://www.libvirt.org/html/libvirt-libvirt.html.
 * 
 * Known api calls to be missing 
 * LIBVIRT_0.1.0 
 * virDefaultErrorFunc 
 * virConnCopyLastError 
 * virFreeError
 * 
 * LIBVIRT_0.4.2 
 * virDomainBlockPeek 
 * virDomainMemoryPeek 
 * 
 * LIBVIRT_0_5.0
 * virEventRegisterImpl 
 * virConnectDomainEventRegister
 * virConnectDomainEventDeregister 
 * 
 * LIBVIRT_0.6.0 
 * virConnectRef 
 * virDomainRef
 * virNetworkRef 
 * virStoragePoolRef 
 * virStorageVolRef 
 * virNodeDeviceRef
 * 
 * LIBVIRT_0.6.1 
 * virFreeError 
 * virSaveLastError 
 * virDomainGetSecurityLabel;
 * virNodeGetSecurityModel; 
 * 
 * LIBVIRT_0.6.4 
 * virInterfaceRef
 * 
 * LIBVIRT_0.7.1 
 * virSecretRef 
 *  
 * LIBVIRT_0.7.2 
 * virStreamRef
 * 
 * LIBVIRT_0.8.0 
 * virNWFilterRef
 *  
 */
public interface Libvirt extends Library {
	
    // Callbacks
    /**
     * Callback interface for authorization
     */
    interface VirConnectAuthCallback extends Callback {
        public int authCallback( final virConnectCredential cred, final int ncred, final Pointer cbdata );
    }

    /**
     * Error callback
     */
    interface VirErrorCallback extends Callback {
        public void errorCallback( final Pointer userData, final virError error );
    }
    
    /**
     * Stream callbacks
     */
    interface VirStreamSinkFunc extends Callback {
        public int sinkCallback( final StreamPointer virStreamPtr, final String data, final NativeLong nbytes, final Pointer opaque );
    }
    
    interface VirStreamSourceFunc extends Callback {
        public int sourceCallback( final StreamPointer virStreamPtr, final String data, final NativeLong nbytes, final Pointer opaque );
    }    
    
    interface VirStreamEventCallback extends Callback {
        public void eventCallback( final StreamPointer virStreamPointer, final int events, final Pointer opaque );
    }
    
    /** 
     * Generic Callbacks
     */
    interface VirFreeCallback extends Callback {
        public void freeCallback( final Pointer opaque );
    }    
    
    interface VirConnectDomainEventGenericCallback extends Callback {
        public void eventCallback( final ConnectionPointer virConnectPtr, final DomainPointer virDomainPointer, final Pointer opaque );
    }
    
    final Libvirt INSTANCE = ( Libvirt )Native.loadLibrary( "virt", Libvirt.class );

    // Constants we need
    public static int VIR_UUID_BUFLEN = 16;
    public static int VIR_UUID_STRING_BUFLEN = (36 + 1);
    public static int VIR_DOMAIN_SCHED_FIELD_LENGTH = 80;

    // Connection Functions
    public String virConnectBaselineCPU( final ConnectionPointer virConnectPtr, final String[] xmlCPUs, final int ncpus, final int flags );
    public int virConnCopyLastError( final ConnectionPointer virConnectPtr, final virError to );
    public int virConnectClose( final ConnectionPointer virConnectPtr );
    public int virConnectCompareCPU( final ConnectionPointer virConnectPtr, final String xmlDesc, final int flags );
    public int virConnectDomainEventRegisterAny( final ConnectionPointer virConnectPtr, final DomainPointer virDomainPtr, final int eventID, final Libvirt.VirConnectDomainEventGenericCallback cb, final Pointer opaque, final Libvirt.VirFreeCallback freecb );
    public int virConnectDomainEventDeregisterAny( final ConnectionPointer virConnectPtr, final int callbackID );
    public void virConnSetErrorFunc( final ConnectionPointer virConnectPtr, final Pointer userData, final VirErrorCallback callback );
    public int virConnectIsEncrypted( final ConnectionPointer virConnectPtr );
    public int virConnectIsSecure( final ConnectionPointer virConnectPtr );    
    public String virConnectFindStoragePoolSources( final ConnectionPointer virConnectPtr, final String type, final String srcSpec, final int flags );
    public String virConnectGetCapabilities( final ConnectionPointer virConnectPtr );
    public String virConnectGetHostname( final ConnectionPointer virConnectPtr );
    public int virConnectGetLibVersion( final ConnectionPointer virConnectPtr, final LongByReference libVer );
    public int virConnectGetMaxVcpus( final ConnectionPointer virConnectPtr, final String type );
    public String virConnectGetType( final ConnectionPointer virConnectPtr );
    public String virConnectGetURI( final ConnectionPointer virConnectPtr );
    public int virConnectGetVersion( final ConnectionPointer virConnectPtr, final LongByReference hvVer );
    public int virConnectListDefinedDomains( final ConnectionPointer virConnectPtr, final String[] name, final int maxnames );
    public int virConnectListDefinedNetworks( final ConnectionPointer virConnectPtr, final String[] name, final int maxnames );
    public int virConnectListDefinedStoragePools( final ConnectionPointer virConnectPtr, final String[] names, final int maxnames );
    public int virConnectListDefinedInterfaces( final ConnectionPointer virConnectPtr, final String[] name, final int maxNames );
    public int virConnectListDomains( final ConnectionPointer virConnectPtr, final int[] ids, final int maxnames );
    public int virConnectListInterfaces( final ConnectionPointer virConnectPtr, final String[] name, final int maxNames );
    public int virConnectListNetworks( final ConnectionPointer virConnectPtr, final String[] name, final int maxnames );
    public int virConnectListNWFilters( final ConnectionPointer virConnectPtr, final String[] name, final int maxnames );
    public int virConnectListSecrets( final ConnectionPointer virConnectPtr, final String[] uids, final int maxUids );
    public int virConnectListStoragePools( final ConnectionPointer virConnectPtr, final String[] names, final int maxnames );
    public int virConnectNumOfDefinedDomains( final ConnectionPointer virConnectPtr );
    public int virConnectNumOfDefinedNetworks( final ConnectionPointer virConnectPtr );
    public int virConnectNumOfDefinedInterfaces( final ConnectionPointer virConnectPtr );
    public int virConnectNumOfDefinedStoragePools( final ConnectionPointer virConnectPtr );
    public int virConnectNumOfDomains( final ConnectionPointer virConnectPtr );
    public int virConnectNumOfInterfaces( final ConnectionPointer virConnectPtr );
    public int virConnectNumOfNetworks( final ConnectionPointer virConnectPtr );
    public int virConnectNumOfNWFilters( final ConnectionPointer virConnectPtr );
    public int virConnectNumOfSecrets( final ConnectionPointer virConnectPtr );      
    public int virConnectNumOfStoragePools( final ConnectionPointer virConnectPtr );
    public ConnectionPointer virConnectOpen( final String name );
    public ConnectionPointer virConnectOpenAuth( final String name, final virConnectAuth auth, final int flags );
    public ConnectionPointer virConnectOpenReadOnly( final String name );
    public virError virConnGetLastError( final ConnectionPointer virConnectPtr );
    public int virConnResetLastError( final ConnectionPointer virConnectPtr );
    public String virConnectDomainXMLFromNative( final ConnectionPointer virConnectPtr, final String nativeFormat,
            final String nativeConfig, final int flags );
    public String virConnectDomainXMLToNative( final ConnectionPointer virConnectPtr, final String nativeFormat, final String domainXML,
            final int flags );
    
    // Global functions
    public int virGetVersion( final LongByReference libVer, final String type, final LongByReference typeVer );
    public int virInitialize();
    public int virCopyLastError( final virError error );
    public virError virGetLastError();
    public void virResetLastError();
    public void virSetErrorFunc( final Pointer userData, final VirErrorCallback callback );    

    // Domain functions
    public int virDomainAbortJob( final DomainPointer virDomainPtr );
    public int virDomainAttachDevice( final DomainPointer virDomainPtr, final String deviceXML );
    public int virDomainAttachDeviceFlags( final DomainPointer virDomainPtr, final String deviceXML, final int flags );    
    public int virDomainBlockStats( final DomainPointer virDomainPtr, final String path, final virDomainBlockStats stats, final int size );
    public int virDomainCoreDump( final DomainPointer virDomainPtr, final String to, final int flags );
    public int virDomainCreate( final DomainPointer virDomainPtr );
    public DomainPointer virDomainCreateLinux( final ConnectionPointer virConnectPtr, final String xmlDesc, final int flags );
    public DomainPointer virDomainCreateXML( final ConnectionPointer virConnectPtr, final String xmlDesc, final int flags );
    public DomainPointer virDomainDefineXML( final ConnectionPointer virConnectPtr, final String xmlDesc );
    public int virDomainDestroy( final DomainPointer virDomainPtr );
    public int virDomainDetachDevice( final DomainPointer virDomainPtr, final String deviceXML );
    public int virDomainDetachDeviceFlags( final DomainPointer virDomainPtr, final String deviceXML, final int flags );    
    public int virDomainFree( final DomainPointer virDomainPtr );
    public int virDomainGetAutostart( final DomainPointer virDomainPtr, final IntByReference value );
    public ConnectionPointer virDomainGetConnect( final DomainPointer virDomainPtr );  
    public int virDomainGetBlockInfo( final DomainPointer virDomainPtr, final String path, final virDomainBlockInfo info, final int flags );
    public int virDomainGetID( final DomainPointer virDomainPtr );
    public int virDomainGetInfo( final DomainPointer virDomainPtr, final virDomainInfo vInfo );
    public int virDomainGetJobInfo( final DomainPointer virDomainPtr, final virDomainJobInfo vInfo );
    public NativeLong virDomainGetMaxMemory( final DomainPointer virDomainPtr );
    public int virDomainGetMaxVcpus( final DomainPointer virDomainPtr );
    public String virDomainGetName( final DomainPointer virDomainPtr );
    public String virDomainGetOSType( final DomainPointer virDomainPtr );
    public int virDomainGetSchedulerParameters( final DomainPointer virDomainPtr, final virSchedParameter[] params,
            final IntByReference nparams );
    public String virDomainGetSchedulerType( final DomainPointer virDomainPtr, final IntByReference nparams );
    public int virDomainGetUUID( final DomainPointer virDomainPtr, final byte[] uuidString );
    public int virDomainGetUUIDString( final DomainPointer virDomainPtr, final byte[] uuidString );
    public int virDomainGetVcpus( final DomainPointer virDomainPtr, final virVcpuInfo[] info, final int maxInfo, final byte[] cpumaps, final int maplen );
    public String virDomainGetXMLDesc( final DomainPointer virDomainPtr, final int flags );
    public int virDomainHasCurrentSnapshot( final DomainPointer virDomainPtr, final int flags );
    public int virDomainHasManagedSaveImage( final DomainPointer virDomainPtr, final int flags );    
    public int virDomainInterfaceStats( final DomainPointer virDomainPtr, final String path, final virDomainInterfaceStats stats, final int size );
    public int virDomainIsActive( final DomainPointer virDomainPtr );
    public int virDomainIsPersistent( final DomainPointer virDomainPtr );
    public DomainPointer virDomainLookupByID( final ConnectionPointer virConnectPtr, final int id );
    public DomainPointer virDomainLookupByName( final ConnectionPointer virConnectPtr, final String name );
    public DomainPointer virDomainLookupByUUID( final ConnectionPointer virConnectPtr, final byte[] uuidBytes );
    public DomainPointer virDomainLookupByUUIDString( final ConnectionPointer virConnectPtr, final String uuidstr );
    public int virDomainManagedSave( final DomainPointer virDomainPtr, final int flags );
    public int virDomainManagedSaveRemove( final DomainPointer virDomainPtr, final int flags );    
    public DomainPointer virDomainMigrate( final DomainPointer virDomainPtr, final ConnectionPointer virConnectPtr,
            final NativeLong flags, final String dname, final String uri, final NativeLong bandwidth );
    public int virDomainMigrateSetMaxDowntime( final DomainPointer virDomainPtr, final long downtime, final int flags );    
    public int virDomainMigrateToURI( final DomainPointer virDomainPtr, final String duri, 
            final NativeLong flags, final String dname, final NativeLong bandwidth );
    public int virDomainMemoryStats( final DomainPointer virDomainPtr, final virDomainMemoryStats[] stats, final int nr_stats, final int flags );
    public int virDomainPinVcpu( final DomainPointer virDomainPtr, final int vcpu, final byte[] cpumap, final int maplen );
    public int virDomainReboot( final DomainPointer virDomainPtr, final int flags );
    public int virDomainRestore( final ConnectionPointer virConnectPtr, final String from );
    public int virDomainRevertToSnapshot( final DomainSnapshotPointer virDomainSnapshotPtr, final int flags );
    public int virDomainResume( final DomainPointer virDomainPtr );
    public int virDomainSave( final DomainPointer virDomainPtr, final String to );
    public int virDomainSetAutostart( final DomainPointer virDomainPtr, final int autoStart );
    public int virDomainSetMaxMemory( final DomainPointer virDomainPtr, final NativeLong maxMemory );
    public int virDomainSetMemory( final DomainPointer virDomainPtr, final NativeLong maxMemory );
    public int virDomainSetSchedulerParameters( final DomainPointer virDomainPtr, final virSchedParameter[] params, final int nparams );
    public int virDomainSetVcpus( final DomainPointer virDomainPtr, final int nvcpus );
    public int virDomainShutdown( final DomainPointer virDomainPtr );
    public int virDomainSuspend( final DomainPointer virDomainPtr );
    public int virDomainUpdateDeviceFlags( final DomainPointer virDomainPtr, final String xml, final int flags );
    public int virDomainUndefine( final DomainPointer virDomainPtr );

    // Network functions
    public ConnectionPointer virNetworkGetConnect( final NetworkPointer virnetworkPtr );
    public int virNetworkCreate( final NetworkPointer virConnectPtr );
    public NetworkPointer virNetworkCreateXML( final ConnectionPointer virConnectPtr, final String xmlDesc );
    public NetworkPointer virNetworkDefineXML( final ConnectionPointer virConnectPtr, final String xmlDesc );
    public int virNetworkDestroy( final NetworkPointer virConnectPtr );
    public int virNetworkFree( final NetworkPointer virConnectPtr );
    public int virNetworkGetAutostart( final NetworkPointer virNetworkPtr, final IntByReference value );
    public String virNetworkGetBridgeName( final NetworkPointer virNetworkPtr );
    public String virNetworkGetName( final NetworkPointer virNetworkPtr );
    public int virNetworkGetUUID( final NetworkPointer virNetworkPtr, final byte[] uuidString );
    public int virNetworkGetUUIDString( final NetworkPointer virNetworkPtr, final byte[] uuidString );
    public String virNetworkGetXMLDesc( final NetworkPointer virNetworkPtr, final int flags );
    public int virNetworkIsActive( final NetworkPointer virNetworkPtr );
    public int virNetworkIsPersistent( final NetworkPointer virNetworkPtr );       
    public NetworkPointer virNetworkLookupByName( final ConnectionPointer virConnectPtr, final String name );
    public NetworkPointer virNetworkLookupByUUID( final ConnectionPointer virConnectPtr, final byte[] uuidBytes );
    public NetworkPointer virNetworkLookupByUUIDString( final ConnectionPointer virConnectPtr, final String uuidstr );
    public int virNetworkSetAutostart( final NetworkPointer virConnectPtr, final int autoStart );
    public int virNetworkUndefine( final NetworkPointer virConnectPtr );

    // Node functions
    public int virNodeGetInfo( final ConnectionPointer virConnectPtr, final virNodeInfo virNodeInfo );
    public int virNodeGetCellsFreeMemory( final ConnectionPointer virConnectPtr, final LongByReference freeMems, final int startCell,
            final int maxCells );
    public long virNodeGetFreeMemory( final ConnectionPointer virConnectPtr );

    // Node/Device functions
    public int virNodeNumOfDevices( final ConnectionPointer virConnectPtr, final String capabilityName, final int flags );
    public int virNodeListDevices( final ConnectionPointer virConnectPtr, final String capabilityName, final String[] names, final int maxnames,
            final int flags );
    public DevicePointer virNodeDeviceLookupByName( final ConnectionPointer virConnectPtr, final String name );
    public String virNodeDeviceGetName( final DevicePointer virDevicePointer );
    public String virNodeDeviceGetParent( final DevicePointer virDevicePointer );
    public int virNodeDeviceNumOfCaps( final DevicePointer virDevicePointer );
    public int virNodeDeviceListCaps( final DevicePointer virDevicePointer, final String[] names, final int maxNames );
    public String virNodeDeviceGetXMLDesc( final DevicePointer virDevicePointer );
    public int virNodeDeviceFree( final DevicePointer virDevicePointer );
    public int virNodeDeviceDettach( final DevicePointer virDevicePointer );
    public int virNodeDeviceReAttach( final DevicePointer virDevicePointer );
    public int virNodeDeviceReset( final DevicePointer virDevicePointer );
    public DevicePointer virNodeDeviceCreateXML( final ConnectionPointer virConnectPtr, final String xml, final int flags );
    public int virNodeDeviceDestroy( final DevicePointer virDevicePointer );

    // Storage Pool
    public int virStoragePoolBuild( final StoragePoolPointer storagePoolPtr, final int flags );
    public int virStoragePoolCreate( final StoragePoolPointer storagePoolPtr, final int flags );
    public StoragePoolPointer virStoragePoolCreateXML( final ConnectionPointer virConnectPtr, final String xml, final int flags );
    public StoragePoolPointer virStoragePoolDefineXML( final ConnectionPointer virConnectPtr, final String xml, final int flags );
    public int virStoragePoolDelete( final StoragePoolPointer storagePoolPtr, final int flags );
    public int virStoragePoolDestroy( final StoragePoolPointer storagePoolPtr );
    public int virStoragePoolFree( final StoragePoolPointer storagePoolPtr );
    public int virStoragePoolGetAutostart( final StoragePoolPointer storagePoolPtr, final IntByReference value );
    public int virStoragePoolGetInfo( final StoragePoolPointer storagePoolPtr, final virStoragePoolInfo info );
    public String virStoragePoolGetName( final StoragePoolPointer storagePoolPtr );
    public int virStoragePoolGetUUID( final StoragePoolPointer storagePoolPtr, final byte[] uuidString );
    public int virStoragePoolGetUUIDString( final StoragePoolPointer storagePoolPtr, final byte[] uuidString );
    public String virStoragePoolGetXMLDesc( final StoragePoolPointer storagePoolPtr, final int flags );
    public int virStoragePoolListVolumes( final StoragePoolPointer storagePoolPtr, final String[] names, final int maxnames );
    public int virStoragePoolIsActive( final StoragePoolPointer storagePoolPtr );
    public int virStoragePoolIsPersistent( final StoragePoolPointer storagePoolPtr );      
    public StoragePoolPointer virStoragePoolLookupByName( final ConnectionPointer virConnectPtr, final String name );
    public StoragePoolPointer virStoragePoolLookupByUUID( final ConnectionPointer virConnectPtr, final byte[] uuidBytes );
    public StoragePoolPointer virStoragePoolLookupByUUIDString(ConnectionPointer virConnectPtr, String uuidstr);
    public StoragePoolPointer virStoragePoolLookupByVolume( final StorageVolPointer storageVolPtr );
    public int virStoragePoolNumOfVolumes( final StoragePoolPointer storagePoolPtr );
    public int virStoragePoolRefresh( final StoragePoolPointer storagePoolPtr );
    public int virStoragePoolSetAutostart( final StoragePoolPointer storagePoolPtr, final int autostart );
    public int virStoragePoolUndefine( final StoragePoolPointer storagePoolPtr );

    // Storage Vol
    public StorageVolPointer virStorageVolCreateXML( final StoragePoolPointer storagePoolPtr, final String xml, final int flags );
    public StorageVolPointer virStorageVolCreateXMLFrom( final StoragePoolPointer storagePoolPtr, final String xml,
            final StorageVolPointer cloneVolume, final int flags );
    public int virStorageVolDelete( final StorageVolPointer storageVolPtr, final int flags );
    public int virStorageVolFree( final StorageVolPointer storageVolPtr );
    public int virStorageVolGetInfo( final StorageVolPointer storageVolPtr, final virStorageVolInfo info );
    public String virStorageVolGetKey( final StorageVolPointer storageVolPtr );
    public String virStorageVolGetName( final StorageVolPointer storageVolPtr );
    public String virStorageVolGetPath( final StorageVolPointer storageVolPtr );
    public String virStorageVolGetXMLDesc( final StorageVolPointer storageVolPtr, final int flags );
    public StorageVolPointer virStorageVolLookupByKey( final ConnectionPointer virConnectPtr, final String name );
    public StorageVolPointer virStorageVolLookupByName( final StoragePoolPointer storagePoolPtr, final String name );
    public StorageVolPointer virStorageVolLookupByPath( final ConnectionPointer virConnectPtr, final String path );
    public int virStorageVolWipe( final StorageVolPointer storageVolPtr, final int flags );

    // Interface Methods
    public int virInterfaceCreate( final InterfacePointer virDevicePointer );
    public InterfacePointer virInterfaceDefineXML( final ConnectionPointer virConnectPtr, final String xml, final int flags );    
    public int virInterfaceDestroy( final InterfacePointer virDevicePointer );    
    public int virInterfaceFree( final InterfacePointer virDevicePointer );    
    public String virInterfaceGetName( final InterfacePointer virInterfacePtr );
    public String virInterfaceGetMACString( final InterfacePointer virInterfacePtr );
    public String virInterfaceGetXMLDesc( final InterfacePointer virInterfacePtr, final int flags );
    public int virInterfaceIsActive( final InterfacePointer virDevicePointer );
    public InterfacePointer virInterfaceLookupByMACString( final ConnectionPointer virConnectPtr, final String mac );
    public InterfacePointer virInterfaceLookupByName( final ConnectionPointer virConnectPtr, final String name );    
    public int virInterfaceUndefine( final InterfacePointer virDevicePointer );
    
    // Secret Methods
    public ConnectionPointer virSecretGetConnect( final SecretPointer virSecretPtr );
    public int virSecretFree( final SecretPointer virSecretPtr );
    public SecretPointer virSecretDefineXML( final ConnectionPointer virConnectPtr, final String xml, final int flags );
    public int virSecretGetUUID( final SecretPointer virSecretPtr, final byte[] uuidString );
    public int virSecretGetUUIDString( final SecretPointer virSecretPtr, final byte[] uuidString );
    public String virSecretGetUsageID( final SecretPointer virSecretPtr );   
    public String virSecretGetValue( final SecretPointer virSecretPtr, final NativeLong value_size, final int flags );      
    public String virSecretGetXMLDesc( final SecretPointer virSecretPtr, final int flags );      
    public SecretPointer virSecretLookupByUsage( final ConnectionPointer virConnectPtr, final int usageType, final String usageID );
    public SecretPointer virSecretLookupByUUID( final ConnectionPointer virConnectPtr, final byte[] uuidBytes );
    public SecretPointer virSecretLookupByUUIDString( final ConnectionPointer virConnectPtr, final String uuidstr );
    public int virSecretSetValue( final SecretPointer virSecretPtr, final String value, final NativeLong value_size, final int flags );        
    public int virSecretUndefine( final SecretPointer virSecretPtr );
    
    //Stream Methods
    public int virStreamAbort( final StreamPointer virStreamPtr );    
    public int virStreamEventAddCallback( final StreamPointer virStreamPtr, final int events, final Libvirt.VirStreamEventCallback cb, 
            final Pointer opaque, final Libvirt.VirFreeCallback ff );
    public int virStreamEventUpdateCallback( final StreamPointer virStreamPtr, final int events );
    public int virStreamEventRemoveCallback( final StreamPointer virStreamPtr );
    public int virStreamFinish( final StreamPointer virStreamPtr );
    public int virStreamFree( final StreamPointer virStreamPtr );
    public StreamPointer virStreamNew( final ConnectionPointer virConnectPtr, final int flags );
    public int virStreamSend( final StreamPointer virStreamPtr, final String data, final NativeLong size );
    public int virStreamSendAll( final StreamPointer virStreamPtr, final Libvirt.VirStreamSourceFunc handler, final Pointer opaque );    
    public int virStreamRecv( final StreamPointer virStreamPtr, final byte[] data, final NativeLong length );
    public int virStreamRecvAll( final StreamPointer virStreamPtr, final Libvirt.VirStreamSinkFunc handler, final Pointer opaque );
    
    //DomainSnapshot Methods
    public DomainSnapshotPointer virDomainSnapshotCreateXML( final DomainPointer virDomainPtr, final String xmlDesc, final int flags );
    public DomainSnapshotPointer virDomainSnapshotCurrent( final DomainPointer virDomainPtr, final int flags );
    public int virDomainSnapshotDelete( final DomainSnapshotPointer virDomainSnapshotPtr, final int flags );
    public String virDomainSnapshotGetXMLDesc( final DomainSnapshotPointer virDomainSnapshotPtr, final int flags );
    public int virDomainSnapshotFree( final DomainSnapshotPointer virDomainSnapshotPtr );
    public int virDomainSnapshotListNames( final DomainPointer virDomainPtr, final String[] names, final int nameslen, final int flags );
    public DomainSnapshotPointer virDomainSnapshotLookupByName( final DomainPointer virDomainPtr, final String name, final int flags );
    public int virDomainSnapshotNum( final DomainPointer virDomainPtr, final int flags );
    
    // Network Filter Methods
    public String virNWFilterGetXMLDesc( final NetworkFilterPointer virNWFilterPtr, final int flags );
    public NetworkFilterPointer virNWFilterDefineXML( final ConnectionPointer virConnectPtr, final String xml );
    public int virNWFilterFree( final NetworkFilterPointer virNWFilterPtr );
    public NetworkFilterPointer virNWFilterLookupByName( final ConnectionPointer virConnectPtr, final String name );
    public NetworkFilterPointer virNWFilterLookupByUUID( final ConnectionPointer virConnectPtr, final byte[] uuidBytes );
    public NetworkFilterPointer virNWFilterLookupByUUIDString( final ConnectionPointer virConnectPtr, final String uuidstr );
    public String virNWFilterGetName( final NetworkFilterPointer virNWFilterPtr );
    public int virNWFilterGetUUID( final NetworkFilterPointer virNWFilterPtr, final byte[] uuidString );
    public int virNWFilterGetUUIDString( final NetworkFilterPointer virNWFilterPtr, final byte[] uuidString );
    public int virNWFilterUndefine( final NetworkFilterPointer virNWFilterPtr );
    
}