package org.libvirt;

import org.libvirt.jna.Libvirt;
import org.libvirt.jna.StreamPointer;

import com.sun.jna.NativeLong;

public final class Stream {

    public static int VIR_STREAM_NONBLOCK = ( 1 << 0 );

    /**
     * the native virStreamPtr.
     */
    StreamPointer VSP;

    /**
     * The Connect Object that represents the Hypervisor of this Domain
     */
    private Connect virConnect;

    /**
     * The libvirt connection from the hypervisor
     */
    protected Libvirt libvirt;

    /**
     * Constructor default
     */
    Stream() {
    }
    
    /**
     * Constructor with arguments
     * 
     * @param virConnect Connect
     * @param VSP StreamPointer
     */
    Stream( final Connect virConnect, final StreamPointer VSP ) {
        this.virConnect = virConnect;
        this.VSP = VSP;
        this.libvirt = virConnect.libvirt;
    }

    /**
     * Request that the in progress data transfer be cancelled abnormally before
     * the end of the stream has been reached.
     * 
     * @return int
     */
    public int abort() throws LibvirtException {
        final int returnValue = libvirt.virStreamAbort( VSP );
        processError();
        return returnValue;
    }

    /**
     * Register a callback to be notified when a stream becomes writable, or
     * readable.
     * 
     * @param events int  The events to monitor
     * @param cb Libvirt.VirStreamEventCallback  The callback method
     * 
     * @return int  0 for success, -1 for failure
     * @throws LibvirtException
     * @see <a
     *      href="http://www.libvirt.org/html/libvirt-libvirt.html#virStreamEventAddCallback">Libvirt
     *      Docs</a>
     */
    public int addCallback( final int events, final Libvirt.VirStreamEventCallback cb ) throws LibvirtException {
        final int returnValue = libvirt.virStreamEventAddCallback( VSP, events, cb, null, null );
        processError();
        return returnValue;
    }

    @Override
    public void finalize() throws LibvirtException {
        free();
    }

    /**
     * Indicate that there is no further data is to be transmitted on the
     * stream.
     * 
     * @return int  0 if success, -1 if failure
     * @throws LibvirtException
     */
    public int finish() throws LibvirtException {
        final int returnValue = libvirt.virStreamFinish( VSP );
        processError();
        return returnValue;
    }

    /**
     * Decrement the reference count on a stream, releasing the stream object if
     * the reference count has hit zero.
     * 
     * @return int  0 on success, or -1 on error.
     * @throws LibvirtException
     */
    public int free() throws LibvirtException {
        int success = 0;
        if( VSP != null ) {
            success = libvirt.virStreamFree( VSP );
            processError();
            VSP = null;
        }
        return success;
    }

    /**
     * Error handling logic to throw errors. Must be called after every libvirt
     * call.
     */
    protected void processError() throws LibvirtException {
        virConnect.processError();
    }

    /**
     * Receieves data from teh stream into the buffer provided.
     * 
     * @param data byte[]  The put the sata into
     * 
     * @return int  The number of bytes read, -1 on error, -2 if the buffer is empty
     * @throws LibvirtException
     */
    public int receive( final byte[] data ) throws LibvirtException {
        final int returnValue = libvirt.virStreamRecv( VSP, data, new NativeLong( data.length ) );
        processError();
        return returnValue;
    }

    /**
     * Batch receive method
     * 
     * @param handler Libvirt.VirStreamSinkFunc  The callback handler
     * 
     * @return int  0 if successfule, -1 otherwise
     * @throws LibvirtException
     * @see http://www.libvirt.org/html/libvirt-libvirt.html#virStreamRecvAll
     */
    public int receiveAll( final Libvirt.VirStreamSinkFunc handler ) throws LibvirtException {
        final int returnValue = libvirt.virStreamRecvAll( VSP, handler, null );
        processError();
        return returnValue;
    }

    /**
     * Remove an event callback from the stream
     * 
     * @return int  0 for success, -1 for failure
     * @throws LibvirtException
     * @see <a
     *      href="http://www.libvirt.org/html/libvirt-libvirt.html#virStreamEventRemoveCallback">Libvirt
     *      Docs</a>
     */
    public int removeCallback() throws LibvirtException {
        final int returnValue = libvirt.virStreamEventRemoveCallback( VSP );
        processError();
        return returnValue;
    }

    /**
     * Write a series of bytes to the stream.
     * 
     * @param data String  The data to write
     * 
     * @return int  The number of bytes written, -1 on error, -2 if the buffer is full
     * @throws LibvirtException
     */
    public int send( final String data ) throws LibvirtException {
        final int returnValue = libvirt.virStreamSend( VSP, data, new NativeLong( data.length() ) );
        processError();
        return returnValue;
    }

    /**
     * Batch send method
     * 
     * @param handler Libvirt.VirStreamSourceFunc  The callback handler
     * 
     * @return int  0 if successfule, -1 otherwise
     * @throws LibvirtException
     * @see <a
     *      href="http://www.libvirt.org/html/libvirt-libvirt.html#virStreamSendAll">Libvirt
     *      Documentation</a>
     */
    public int sendAll(Libvirt.VirStreamSourceFunc handler) throws LibvirtException {
        final int returnValue = libvirt.virStreamSendAll( VSP, handler, null );
        processError();
        return returnValue;
    }

    /**
     * Changes the set of events to monitor for a stream.
     * 
     * @param events int  The events to monitor
     * 
     * @return int  0 for success, -1 for failure
     * @throws LibvirtException
     * @see <a
     *      href="http://www.libvirt.org/html/libvirt-libvirt.html#virStreamEventUpdateCallback">Libvirt
     *      Docs</a>
     */
    public int updateCallback(int events) throws LibvirtException {
        final int returnValue = libvirt.virStreamEventUpdateCallback( VSP, events );
        processError();
        return returnValue;
    }
    
}