package org.libvirt;

import org.libvirt.jna.Libvirt;
import org.libvirt.jna.StoragePoolPointer;
import org.libvirt.jna.StorageVolPointer;
import org.libvirt.jna.virStoragePoolInfo;

import com.sun.jna.Native;
import com.sun.jna.ptr.IntByReference;

/**
 * A collection of storage
 */
public final class StoragePool {

    static final class BuildFlags {
    	
        /**
         * Regular build from scratch
         */
        static final int VIR_STORAGE_POOL_BUILD_NEW = 0;
        
        /**
         * Repair / reinitialize
         */
        static final int VIR_STORAGE_POOL_BUILD_REPAIR = 1;
        
        /**
         * Extend existing pool
         */
        static final int VIR_STORAGE_POOL_BUILD_RESIZE = 2;
    }

    static final class DeleteFlags {
    	
        /**
         * Delete metadata only (fast)
         */
        static final int VIR_STORAGE_POOL_DELETE_NORMAL = 0;
        
        /**
         * Clear all data to zeros (slow)
         */
        static final int VIR_STORAGE_POOL_DELETE_ZEROED = 1;
    }

    /**
     * the native virStoragePoolPtr.
     */
    protected StoragePoolPointer VSPP;

    /**
     * The VirConnect Object that represents the Hypervisor of this Domain
     */
    protected Connect virConnect;

    /**
     * the libvirt instance
     */
    protected Libvirt libvirt;

    /**
     * Constrcutor default
     */
    StoragePool() {
    }
    
    /**
     * Constructs a VirStoragePool object from a known native virStoragePoolPtr,
     * and a VirConnect object. For use when native libvirt returns a
     * virStoragePoolPtr, i.e. error handling.
     * 
     * @param virConnect Connect  The Domain's hypervisor
     * @param VSPP StoragePoolPointer  The native virStoragePoolPtr
     */
    StoragePool( final Connect virConnect, final StoragePoolPointer VSPP ) {
        this.virConnect = virConnect;
        this.VSPP = VSPP;
        this.libvirt = virConnect.libvirt;
    }

    /**
     * Build the underlying storage pool
     * 
     * @param flags int  Future flags, use 0 for now
     */
    public void build( final int flags ) throws LibvirtException {
        libvirt.virStoragePoolBuild( VSPP, flags );
        processError();
    }

    /**
     * Starts this inactive storage pool
     * 
     * @param flags int  Future flags, use 0 for now
     */
    public void create( final int flags ) throws LibvirtException {
        libvirt.virStoragePoolCreate( VSPP, flags );
        processError();
    }

    /**
     * Delete the underlying pool resources. This is a non-recoverable
     * operation. The virStoragePool object itself is not free'd.
     * 
     * @param flags int  Flags for obliteration process
     */
    public void delete( final int flags ) throws LibvirtException {
        libvirt.virStoragePoolDelete( VSPP, flags );
        processError();
    }

    /**
     * Destroy an active storage pool. This will deactivate the pool on the
     * host, but keep any persistent config associated with it. If it has a
     * persistent config it can later be restarted with virStoragePoolCreate().
     * This does not free the associated virStoragePoolPtr object.
     */
    public void destroy() throws LibvirtException {
        libvirt.virStoragePoolDestroy( VSPP );
        processError();
    }

    @Override
    public void finalize() throws LibvirtException {
        free();
    }

    /**
     * Free a storage pool object, releasing all memory associated with it. Does
     * not change the state of the pool on the host.
     * 
     * @return number of references left (>= 0) for success, -1 for failure.
     * @throws LibvirtException
     */
    public int free() throws LibvirtException {
        int success = 0;
        if( VSPP != null ) {
            success = libvirt.virStoragePoolFree( VSPP );
            processError();
            VSPP = null;
        }
        return success;
    }

    /**
     * Fetches the value of the autostart flag, which determines whether the
     * pool is automatically started at boot time
     * 
     * @return boolean  The result
     * @throws LibvirtException
     */
    public boolean getAutostart() throws LibvirtException {
        final IntByReference autoStart = new IntByReference();
        libvirt.virStoragePoolGetAutostart( VSPP, autoStart );
        processError();
        return ( ( autoStart.getValue() != 0 ) ? true : false );
    }

    /**
     * Provides the connection pointer associated with a storage pool.
     * 
     * @return Connect  The Connect object
     */
    public Connect getConnect() {
        return virConnect;
    }

    /**
     * Get volatile information about the storage pool such as free space /
     * usage summary
     * 
     * @return StoragePoolInfo  A StoragePoolInfo object describing this storage pool
     * @throws LibvirtException
     */
    public StoragePoolInfo getInfo() throws LibvirtException {
        final virStoragePoolInfo vInfo = new virStoragePoolInfo();
        libvirt.virStoragePoolGetInfo( VSPP, vInfo );
        processError();
        return new StoragePoolInfo( vInfo );
    }

    /**
     * Fetch the locally unique name of the storage pool
     * 
     * @return String  The name
     * @throws LibvirtException
     */
    public String getName() throws LibvirtException {
        final String returnValue = libvirt.virStoragePoolGetName( VSPP );
        processError();
        return returnValue;
    }

    /**
     * Fetch the globally unique ID of this storage pool
     * 
     * @return int[]  The UUID as an unpacked int array
     * @throws LibvirtException
     */
    public int[] getUUID() throws LibvirtException {
        final byte[] bytes = new byte[ Libvirt.VIR_UUID_BUFLEN ];
        final int success = libvirt.virStoragePoolGetUUID( VSPP, bytes );
        processError();
        int[] returnValue = new int[ 0 ];
        if( success == 0 ) {
            returnValue = Connect.convertUUIDBytes( bytes );
        }
        return returnValue;
    }

    /**
     * Fetch the globally unique ID of the storage pool as a string
     * 
     * @return String  The UUID in canonical String format
     * @throws LibvirtException
     */
    public String getUUIDString() throws LibvirtException {
        final byte[] bytes = new byte[ Libvirt.VIR_UUID_STRING_BUFLEN ];
        final int success = libvirt.virStoragePoolGetUUIDString( VSPP, bytes );
        processError();
        String returnValue = null;
        if( success == 0 ) {
            returnValue = Native.toString( bytes );
        }
        return returnValue;
    }

    /**
     * Fetch an XML document describing all aspects of the storage pool. This is
     * suitable for later feeding back into the virStoragePoolCreateXML method.
     * 
     * @param flags int  Flags for XML format options (set of virDomainXMLFlags)
     * 
     * @return String  A XML document -java @throws LibvirtException
     */
    public String getXMLDesc( final int flags ) throws LibvirtException {
        final String returnValue = libvirt.virStoragePoolGetXMLDesc( VSPP, flags );
        processError();
        return returnValue;
    }

    /**
     * Determine if the storage pool is currently running
     * 
     * @return int  1 if running, 0 if inactive, -1 on error
     * @throws LibvirtException
     * @see <a
     *      href="http://www.libvirt.org/html/libvirt-libvirt.html#virStoragePoolIsActive">Libvirt
     *      Documentation</a>
     */
    public int isActive() throws LibvirtException {
        final int returnValue = libvirt.virStoragePoolIsActive( VSPP );
        processError();
        return returnValue;
    }

    /**
     * Determine if the storage pool has a persistent configuration which means
     * it will still exist after shutting down
     * 
     * @return int  1 if persistent, 0 if transient, -1 on error
     * @throws LibvirtException
     * @see <a
     *      href="http://www.libvirt.org/html/libvirt-libvirt.html#virStoragePoolIsPersistent">Libvirt
     *      Documentation</a>
     */
    public int isPersistent() throws LibvirtException {
        final int returnValue = libvirt.virStoragePoolIsPersistent( VSPP );
        processError();
        return returnValue;
    }

    /**
     * Fetch list of storage volume names
     * 
     * @return String[]  An Array of Strings that contains the names of the storage
     *                   volumes
     * @throws LibvirtException
     */
    public String[] listVolumes() throws LibvirtException {
        final int num = numOfVolumes();
        final String[] returnValue = new String[ num ];
        libvirt.virStoragePoolListVolumes( VSPP, returnValue, num );
        processError();
        return returnValue;
    }

    /**
     * Fetch the number of storage volumes within a pool
     * 
     * @return int  The number of storage pools
     * @throws LibvirtException
     */
    public int numOfVolumes() throws LibvirtException {
        final int returnValue = libvirt.virStoragePoolNumOfVolumes( VSPP );
        processError();
        return returnValue;
    }

    /**
     * 
     * @throws LibvirtException
     */
    protected void processError() throws LibvirtException {
        virConnect.processError();
    }

    /**
     * Request that the pool refresh its list of volumes. This may involve
     * communicating with a remote server, and/or initializing new devices at
     * the OS layer
     * 
     * @param flags int  Flags to control refresh behaviour (currently unused, use 0)
     * @throws LibvirtException
     */
    public void refresh( final int flags ) throws LibvirtException {
        libvirt.virStoragePoolRefresh( VSPP );
        processError();
    }

    /**
     * Sets the auto start flag
     * 
     * @param autostart int  New flag setting
     * 
     * @throws LibvirtException
     */
    public void setAutostart( final int autostart ) throws LibvirtException {
        libvirt.virStoragePoolSetAutostart( VSPP, autostart );
    }

    /**
     * Create a storage volume within a pool based on an XML description. Not
     * all pools support creation of volumes
     * 
     * @param xmlDesc String  Description of volume to create
     * @param flags int  Flags for creation (unused, pass 0)
     * 
     * @return StorageVol  The storage volume
     * @throws LibvirtException
     */
    public StorageVol storageVolCreateXML( final String xmlDesc, final int flags ) throws LibvirtException {
        final StorageVolPointer sPtr = libvirt.virStorageVolCreateXML( VSPP, xmlDesc, flags );
        processError();
        return new StorageVol( virConnect, sPtr );
    }

    /**
     * Create a storage volume in the parent pool, using the 'clonevol' volume
     * as input. Information for the new volume (name, perms) are passed via a
     * typical volume XML description.
     * 
     * @param xmlDesc String
     * @param cloneVolume StorageVol
     * @param flags int
     * 
     * @return StorageVol
     * @throws LibvirtException
     */
    public StorageVol storageVolCreateXMLFrom( final String xmlDesc, final StorageVol cloneVolume, final int flags )
            throws LibvirtException {
        final StorageVolPointer sPtr = libvirt.virStorageVolCreateXMLFrom( VSPP, xmlDesc, cloneVolume.VSVP, flags );
        processError();
        return new StorageVol( virConnect, sPtr );
    }

    /**
     * Fetch an object representing to a storage volume based on its name within
     * a pool.
     * 
     * @param name String  Name of storage volume
     * 
     * @return StorageVol  The StorageVol object found
     * @throws LibvirtException
     */
    public StorageVol storageVolLookupByName( final String name ) throws LibvirtException {
        final StorageVolPointer sPtr = libvirt.virStorageVolLookupByName( VSPP, name );
        processError();
        return new StorageVol( virConnect, sPtr );
    }

    /**
     * Undefine an inactive storage pool.
     * 
     * @throws LibvirtException
     */
    public void undefine() throws LibvirtException {
        libvirt.virStoragePoolUndefine( VSPP );
        processError();
    }

}