package org.libvirt;

import org.libvirt.jna.Libvirt;
import org.libvirt.jna.SecretPointer;

import com.sun.jna.Native;
import com.sun.jna.NativeLong;

/**
 * A secret defined by libvirt
 */
public final class Secret {

    /**
     * the native virSecretPtr.
     */
    SecretPointer VSP;

    /**
     * The Connect Object that represents the Hypervisor of this Domain
     */
    private Connect virConnect;

    /**
     * The libvirt connection from the hypervisor
     */
    protected Libvirt libvirt;

    /**
     * Constructor default
     */
    Secret() {
    }
    
    /**
     * Constructor with arguments
     * 
     * @param virConnect Connect
     * @param VSP SecretPointer
     */
    Secret( final Connect virConnect, final SecretPointer VSP ) {
        this.virConnect = virConnect;
        this.VSP = VSP;
        this.libvirt = virConnect.libvirt;
    }

    @Override
    public void finalize() throws LibvirtException {
        free();
    }

    /**
     * Release the secret handle. The underlying secret continues to exist.
     * 
     * @return int  0 on success, or -1 on error.
     * @throws LibvirtException
     */
    public int free() throws LibvirtException {
        int success = 0;
        if( VSP != null ) {
            success = libvirt.virSecretFree( VSP );
            processError();
            VSP = null;
        }
        return success;
    }

    /**
     * Get the unique identifier of the object with which this secret is to be
     * used.
     * 
     * @return String  A string identifying the object using the secret, or NULL upon
     *                 error
     * @throws LibvirtException
     */
    public String getUsageID() throws LibvirtException {
        final String returnValue = libvirt.virSecretGetUsageID( VSP );
        processError();
        return returnValue;
    }

    /**
     * Get the UUID for this secret.
     * 
     * @return int[]  The UUID as an unpacked int array
     * @throws LibvirtException
     * @see <a href="http://www.ietf.org/rfc/rfc4122.txt">rfc4122</a>
     */
    public int[] getUUID() throws LibvirtException {
        final byte[] bytes = new byte[ Libvirt.VIR_UUID_BUFLEN ];
        final int success = libvirt.virSecretGetUUID( VSP, bytes );
        processError();
        int[] returnValue = new int[ 0 ];
        if( success == 0 ) {
            returnValue = Connect.convertUUIDBytes( bytes );
        }
        return returnValue;
    }

    /**
     * Gets the UUID for this secret as string.
     * 
     * @return String  The UUID in canonical String format
     * @throws LibvirtException
     * @see <a href="http://www.ietf.org/rfc/rfc4122.txt">rfc4122</a>
     */
    public String getUUIDString() throws LibvirtException {
        final byte[] bytes = new byte[ Libvirt.VIR_UUID_STRING_BUFLEN ];
        final int success = libvirt.virSecretGetUUIDString( VSP, bytes );
        processError();
        String returnValue = null;
        if( success == 0 ) {
            returnValue = Native.toString( bytes );
        }
        return returnValue;
    }

    /**
     * Fetches the value of the secret
     * 
     * @return String  The value of the secret, or null on failure.
     */
    public String getValue() throws LibvirtException {
        final String returnValue = libvirt.virSecretGetValue( VSP, new NativeLong(), 0 );
        processError();
        return returnValue;
    }

    /**
     * Fetches an XML document describing attributes of the secret.
     * 
     * @return String  The XML document
     */
    public String getXMLDesc() throws LibvirtException {
        final String returnValue = libvirt.virSecretGetXMLDesc( VSP, 0 );
        processError();
        return returnValue;
    }

    /**
     * Error handling logic to throw errors. Must be called after every libvirt
     * call.
     */
    protected void processError() throws LibvirtException {
        virConnect.processError();
    }

    /**
     * Sets the value of the secret
     * 
     * @return int  0 on success, -1 on failure.
     */
    public int setValue(String value) throws LibvirtException {
        final int returnValue = libvirt.virSecretSetValue( VSP, value, new NativeLong( value.length() ), 0 );
        processError();
        return returnValue;
    }

    /**
     * Undefines, but does not free, the Secret.
     * 
     * @return int  0 on success, -1 on failure.
     */
    public int undefine() throws LibvirtException {
        final int returnValue = libvirt.virSecretUndefine( VSP );
        processError();
        return returnValue;
    }
    
}