/*----------------------------------------------------------------------------------*/
/* Copyright 2002-2009, Distributed Systems Architecture Group, Universidad   
 * Complutense de Madrid (dsa-research.org)                                   
 *                                                                            
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */ 
/*-----------------------------------------------------------------------------------*/

#include <config.h>
#include <sys/utsname.h>

#include "virterror_internal.h"
#include "one_conf.h"

/* --------------------------------------------------------------------------------- */
/* --------------------------------------------------------------------------------- */
/* --------------------------------------------------------------------------------- */

/**
 * oneCapsInit initialize the driver capabilities
 * @return a pointer to the driver capabilities NULL in case of error
 */

virCapsPtr oneCapsInit(void)
{
    struct utsname  utsname;
    virCapsPtr      caps;
    virCapsGuestPtr guest;

    uname(&utsname);

    if ((caps = virCapabilitiesNew(utsname.machine,0,0)) == NULL)
    {
        goto no_memory;
    }

    virCapabilitiesSetMacPrefix(caps,(unsigned char[]){ 0x52, 0x54, 0x00 });

    if ((guest = virCapabilitiesAddGuest(caps,
                                         "hvm",
                                         utsname.machine,
                                         sizeof(int) == 4 ? 32 : 64,
                                         NULL,
                                         NULL,
                                         0,
                                         NULL)) == NULL)
    {
        goto no_memory;
    }

    if (virCapabilitiesAddGuestDomain(guest,
                                      "hvm",
                                      NULL,
                                      NULL,
                                      0,
                                      NULL) == NULL)
    {
        goto no_memory;
    }

    return caps;

no_memory:

    virCapabilitiesFree(caps);
    return NULL;
}

/* --------------------------------------------------------------------------------- */
/* --------------------------------------------------------------------------------- */
/* --------------------------------------------------------------------------------- */

/**
 * oneSubmitVM generates an OpenNebula description file and submits the new VM
 * @param driver the OpenNebula driver
 * @param vm the virtual machine pointer
 * @return the OpenNebula ID for the new VM or -1 in case of error
 */

int oneSubmitVM(virConnectPtr    conn ATTRIBUTE_UNUSED,
                one_driver_t*    driver,
                virDomainObjPtr  vm)
{	
    FILE* fd=NULL;
    char  path[256];
    int   oneid;

    snprintf(path,256,"/tmp/one-%d",driver->nextid);

    fd=fopen(path,"w");

    if(fd==NULL)
    {
        perror("Creating a temporal description file:");
	return -1;
    }

    if(xmlOneTemplate(vm->def,fd))
    {
        unlink(path);
        fclose(fd);
        return -1;
    }

    fclose(fd);

    if( (oneid=c_oneAllocate(path))<0 )
    {
        oneError(conn, NULL, VIR_ERR_OPERATION_FAILED,
                     "Error submitting virtual machine to OpenNebula");
        unlink(path);             
       	return -1;
    };

    unlink(path);

    return oneid;
};

/* --------------------------------------------------------------------------------- */
/* --------------------------------------------------------------------------------- */
/* --------------------------------------------------------------------------------- */

/**
 * xmlOneTemplate Generate an OpenNebula template to deploy a VM from libvirt
 * internal Domain definition.
 * @param def  Internal libvirt Domain definition
 * @param fd   ONE template file to create 
 * @return 0 if success
 */

int xmlOneTemplate(virDomainDefPtr def, FILE* fd)
{
    char* buf;
    int   i;

    fprintf(fd,"#OpenNebula Template automatically generated by libvirt\n");
    
    fprintf(fd,"NAME = %s\n",def->name);
    fprintf(fd,"CPU = %ld\n",(def->vcpus));
    fprintf(fd,"MEMORY = %ld\n",(def->maxmem)/1024);
	
    /*Optional Booting OpenNebula Information:*/

    if( def->os.kernel )
    {
        fprintf(fd,"OS=[ kernel = \"%s\"",def->os.kernel);

        if(def->os.initrd)
            fprintf(fd,",\n    initrd = \"%s\"",def->os.initrd);
        if(def->os.cmdline)
            fprintf(fd,",\n    kernel_cmd = \"%s\"",def->os.cmdline);
        if(def->os.root)
            fprintf(fd,",\n    root  = \"%s\"",def->os.root);
        
	fprintf(fd," ]\n");
    }

    /* set Disks & NICS */

    for(i=0 ; i<def->ndisks ; i++)
    {
        if ( !def->disks[i] || !def->disks[i]->src || !def->disks[i]->src )
        {
            continue;
        }
	
        fprintf(fd, "DISK=[ source = \"%s\",\n"
                    "       target = \"%s\",\n"
                    "       readonly =",
                    def->disks[i]->src, 
                    def->disks[i]->dst);

        if(def->disks[i]->readonly) 
            fprintf(fd,"\"yes\"]\n");
        else 
            fprintf(fd,"\"no\"]\n");
    }
    
    for(i=0 ; i< def->nnets ; i++)
    {
        if ( !def->nets[i] )
        {
            continue;
        }

        switch(def->nets[i]->type)
        {
            case VIR_DOMAIN_NET_TYPE_BRIDGE:
                fprintf(fd,"NIC=[ bridge =\"%s\",\n",def->nets[i]->data.bridge.brname);
                if(def->nets[i]->ifname)
                {
                    fprintf(fd,"      target =\"%s\",\n",def->nets[i]->ifname);
                }

	        fprintf(fd,"      mac =\"%02x:%02x:%02x:%02x:%02x:%02x\" ]\n",
                        def->nets[i]->mac[0],def->nets[i]->mac[1],
                        def->nets[i]->mac[2],def->nets[i]->mac[3],
                        def->nets[i]->mac[4],def->nets[i]->mac[5]);  
                break;

            case VIR_DOMAIN_NET_TYPE_NETWORK:
                fprintf(fd,"NIC=[ network=\"%s\" ]\n",def->nets[i]->data.network.name);
                break;
        }
    }

    return 0;
};
